<?php

namespace Drupal\github_webhook\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configure GitHub Webhook settings.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['github_webhook.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'github_webhook_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('github_webhook.settings');

    $form['github_token'] = [
      '#type' => 'textfield',
      '#title' => $this->t('GitHub Personal Access Token'),
      '#default_value' => $config->get('github_token') ?? '',
      '#description' => $this->t('GitHub Personal Access Token with repo permissions. Create one at: <a href="@url" target="_blank">@url</a>', [
        '@url' => 'https://github.com/settings/tokens',
      ]),
      '#required' => FALSE,
    ];

    $form['repositories'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('GitHub Repositories'),
      '#description' => $this->t('Add GitHub repositories that should be triggered when content is created or updated. Format: username/repository-name'),
      '#tree' => TRUE,
    ];

    // Get existing repositories from config
    $repositories = $config->get('github_repositories') ?? [''];

    // Ensure we always have at least one empty field
    if (empty($repositories)) {
      $repositories = [''];
    }

    // Handle add more functionality
    $num_repos = $form_state->get('num_repositories');
    if ($num_repos === NULL) {
      $num_repos = count($repositories);
      $form_state->set('num_repositories', $num_repos);
    }

    // Build repository fields
    for ($i = 0; $i < $num_repos; $i++) {
      $form['repositories'][$i] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['repository-row']],
      ];

      $form['repositories'][$i]['repo'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Repository #@num', ['@num' => $i + 1]),
        '#default_value' => $repositories[$i] ?? '',
        '#placeholder' => 'username/repository-name',
        '#size' => 60,
      ];

      if ($num_repos > 1) {
        $form['repositories'][$i]['remove'] = [
          '#type' => 'submit',
          '#value' => $this->t('Remove'),
          '#name' => 'remove_repo_' . $i,
          '#submit' => ['::removeRepository'],
          '#ajax' => [
            'callback' => '::repositoriesCallback',
            'wrapper' => 'repositories-wrapper',
          ],
          '#limit_validation_errors' => [],
          '#repo_index' => $i,
        ];
      }
    }

    $form['repositories']['add_more'] = [
      '#type' => 'submit',
      '#value' => $this->t('Add another repository'),
      '#submit' => ['::addMoreRepository'],
      '#ajax' => [
        'callback' => '::repositoriesCallback',
        'wrapper' => 'repositories-wrapper',
      ],
      '#limit_validation_errors' => [],
    ];

    // Wrap repositories in a container for AJAX
    $form['repositories']['#prefix'] = '<div id="repositories-wrapper">';
    $form['repositories']['#suffix'] = '</div>';

    return parent::buildForm($form, $form_state);
  }

  /**
   * Callback for adding more repositories.
   */
  public function addMoreRepository(array &$form, FormStateInterface $form_state) {
    $num_repos = $form_state->get('num_repositories');
    $form_state->set('num_repositories', $num_repos + 1);
    $form_state->setRebuild();
  }

  /**
   * Callback for removing a repository.
   */
  public function removeRepository(array &$form, FormStateInterface $form_state) {
    $triggering_element = $form_state->getTriggeringElement();
    $repo_index = $triggering_element['#repo_index'];

    // Get current values
    $values = $form_state->getValue('repositories');

    // Remove the repository at the specified index
    unset($values[$repo_index]);
    $values = array_values($values); // Re-index array

    // Update form state
    $form_state->setValue('repositories', $values);
    $num_repos = $form_state->get('num_repositories');
    $form_state->set('num_repositories', $num_repos - 1);
    $form_state->setRebuild();
  }

  /**
   * AJAX callback for repository operations.
   */
  public function repositoriesCallback(array &$form, FormStateInterface $form_state) {
    return $form['repositories'];
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);

    // Validate repository format
    $repositories = $form_state->getValue('repositories');
    if (is_array($repositories)) {
      foreach ($repositories as $index => $repo_data) {
        if (is_array($repo_data) && isset($repo_data['repo'])) {
          $repo = trim($repo_data['repo']);
          if (!empty($repo) && !preg_match('/^[\w\-\.]+\/[\w\-\.]+$/', $repo)) {
            $form_state->setErrorByName("repositories][$index][repo",
              $this->t('Repository must be in format: username/repository-name'));
          }
        }
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Extract and clean repository values
    $repositories = [];
    $repo_values = $form_state->getValue('repositories');

    if (is_array($repo_values)) {
      foreach ($repo_values as $key => $repo_data) {
        // Skip non-numeric keys (like 'add_more')
        if (is_numeric($key) && is_array($repo_data) && isset($repo_data['repo'])) {
          $repo = trim($repo_data['repo']);
          if (!empty($repo)) {
            $repositories[] = $repo;
          }
        }
      }
    }

    // Save configuration
    $this->config('github_webhook.settings')
      ->set('github_token', $form_state->getValue('github_token'))
      ->set('github_repositories', $repositories)
      ->save();

    parent::submitForm($form, $form_state);
  }

}
